/*:
 * @target MZ
 * @plugindesc Dynamisches Lichtsystem für RPG Maker MZ (Fix für Bewegungsfehler)
 * @author Yui
 *
 * @help
 * Dieses Plugin ermöglicht dynamische Lichtquellen auf der Map.
 * 
 * Plugin-Befehle:
 * - SetLight x y radius intensity color
 * - SetGlobalLight intensity
 *
 * @command SetLight
 * @text Lichtquelle setzen
 * @desc Erstellt eine Lichtquelle an einer bestimmten Position.
 *
 * @arg x
 * @text X-Position
 * @type number
 * @desc X-Koordinate der Lichtquelle (in Tiles).
 *
 * @arg y
 * @text Y-Position
 * @type number
 * @desc Y-Koordinate der Lichtquelle (in Tiles).
 *
 * @arg radius
 * @text Radius
 * @type number
 * @desc Radius des Lichts (in Tiles).
 *
 * @arg intensity
 * @text Intensität
 * @type number
 * @desc Lichtintensität (0.0 - 1.0).
 *
 * @arg color
 * @text Lichtfarbe
 * @type string
 * @desc Farbe des Lichts (Hex-Code, z. B. #FFFFFF für Weiß).
 *
 * @command SetGlobalLight
 * @text Globale Lichtstärke setzen
 * @desc Setzt die allgemeine Lichtintensität auf der Karte.
 *
 * @arg intensity
 * @text Intensität
 * @type number
 * @desc Lichtintensität (0.0 - 1.0).
 */

(() => {
    const pluginName = "Yui_DynamicLighting";

    let globalLightIntensity = 1.0; // Standard: volle Helligkeit
    let lights = []; // Liste aller Lichtquellen

    PluginManager.registerCommand(pluginName, "SetLight", args => {
        const x = Number(args.x) * 48; // Map-Koordinaten in Pixel umrechnen
        const y = Number(args.y) * 48;
        const radius = Number(args.radius) * 48;
        const intensity = Number(args.intensity);
        const color = args.color || "#FFFFFF";
        lights.push({ x, y, radius, intensity, color });
    });

    PluginManager.registerCommand(pluginName, "SetGlobalLight", args => {
        globalLightIntensity = Number(args.intensity);
    });

    const _Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
    Spriteset_Map.prototype.createLowerLayer = function() {
        _Spriteset_Map_createLowerLayer.call(this);
        this.createLightingLayer();
    };

    Spriteset_Map.prototype.createLightingLayer = function() {
        this._lightingSprite = new PIXI.Container();
        this.addChild(this._lightingSprite);
    };

    const _Spriteset_Map_update = Spriteset_Map.prototype.update;
    Spriteset_Map.prototype.update = function() {
        _Spriteset_Map_update.call(this);
        this.updateLighting();
    };

    Spriteset_Map.prototype.updateLighting = function() {
        this._lightingSprite.removeChildren();
        
        let darkness = new PIXI.Graphics();
        darkness.beginFill(0x000000, 1 - globalLightIntensity);
        darkness.drawRect(0, 0, Graphics.width, Graphics.height);
        darkness.endFill();
        this._lightingSprite.addChild(darkness);

        lights.forEach(light => {
            let glow = new PIXI.Graphics();
            glow.beginFill(PIXI.utils.string2hex(light.color));
            glow.drawCircle(0, 0, light.radius);
            glow.endFill();
            glow.alpha = light.intensity;
            glow.x = light.x - ($gameMap.displayX() * 48);
            glow.y = light.y - ($gameMap.displayY() * 48);
            this._lightingSprite.addChild(glow);
        });
    };
})();